/*

Audio manager

We have 3 audio channels for different purposes:

		- music/bgm
		.load all bgm in the main preloader as bgm1, bgm2, bgm3 ...
		.play a bgm by calling playBGM(bgmIndex) where bgmIndex is the postfix number of the key

		- sfx (sound effects)
		.load all sfx in the main preloader, using arbitrary keys
		.play a sfx by calling playSFX(soundKey, rate)
		.default for 'rate' is 1 (normal speed)

		- speech
		.the playSound function loads mp3 files on the fly
		.no preload needed, but sound may take a second to load
		.must be MP3 files inside the assets/sounds/ folder

*/ class Audio extends Phaser.Scene {
    constructor(){
        super("Audio");
    }
    init() {
        this.audFile;
        this.soundTimer;
        this.bgm;
        this.sfx;
        this.modal = this.scene.get("Modal");
    }
    preload() {}
    create() {
        var parent = this;
        log("=> launching Audio manager");
        this.game.music.setVolume(0.1);
        this.game.sfx.setVolume(0.1);
        this.game.speech.setVolume(0.5);
    }
    update() {}
    /// BGM
    playBGM(bgmIndex) {
        log(this);
        if (this.bgm) this.bgm.stop();
        this.bgm = this.game.music.add([
            "bgm" + bgmIndex
        ]);
        this.bgm.setRate(1);
        this.bgm.play();
        this.bgm.setLoop(true);
    }
    stopBGM() {
        if (this.bgm != undefined) this.bgm.stop();
    }
    setBGMVolume(newVolume) {
        //this.bgm.setVolume(newVolume);
        this.game.music.setVolume(newVolume);
    }
    playSFX(soundKey, rate = 1) {
        this.sfx = this.game.sfx.add(soundKey);
        this.sfx.setRate(rate);
        this.sfx.play();
    }
    setSFXVolume(newVolume) {
        this.game.sfx.setVolume(newVolume);
    }
    setSpeechVolume(newVolume) {
        this.game.speech.setVolume(newVolume);
    }
    // play a dynamically loaded sound file from assets/sounds/
    // sounds may take a second to load so this is meant for voiceover files
    playSound(soundKey, { loop =false , blocksInputs =false , stopsOther =false  }) {
        if (this.audFile) {
            if (this.audFile.isPlaying && stopsOther) {
                this.audFile.stop();
                clearTimeout(this.soundTimer);
                this.removeSoundKeys(this);
            }
        }
        this.load.audio(soundKey, "./assets/sounds/dialog/" + soundKey + ".mp3");
        this.load.once(Phaser.Loader.Events.COMPLETE, ()=>{
            log("playing sound: ", soundKey);
            this.audFile = this.game.speech.add(soundKey);
            this.audFile.loop = loop;
            this.audFile.setRate(1);
            this.audFile.play();
            if (blocksInputs) this.modal.blockInputs();
            // dynamically loaded sounds do not stop by themselves (as they can be streamed)
            // so we must trigger the event manually after sound finished playing
            if (!loop) this.soundTimer = setTimeout(()=>{
                this.audFile.stop();
            }, Math.ceil(this.audFile.duration) * 999);
            this.audFile.on("stop", (ev)=>{
                clearTimeout(this.soundTimer);
                if (blocksInputs) this.modal.blockInputs(false);
            });
        });
        this.load.start();
    }
    // in case you have a looping sound
    stopSound(key) {
        this.audFile.stop(key);
    }
    stopSounds() {
        this.audFile.stop();
    }
    // before loading a new set of audio files
    // you can clean up the keys/references of unused sounds with this function
    // -> they are going to be garbage collected by the browser in the next gc cycle
    removeSoundKeys() {
        var parent = this;
        var allKeys = this.cache.audio.getKeys();
        allKeys.forEach(function(key) {
            log("removing ", key);
            var cache = parent.cache.text;
            cache.remove(key);
            parent.cache.audio.remove(key);
            parent.sound.remove(key);
        });
    }
}

//# sourceMappingURL=index.d16d9df6.js.map
